/** @file   statistics.h
 * @brief   Declaration of Statistics - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_STATISTICS_H
#define H_WWW_STATISTICS_H

#include <iostream>
#include "objectid.h"


namespace WeWantWar {


/** @class  Statistics
 * @brief   Provides the statistics features.
 * @author  Tomi Lamminsaari
 *
 * 
 */
class Statistics
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** These are the things we keep the statistics. These are used as a
   * parameter when updating or retrieving the statistics.
   */
  enum StatType {
    /** How many rounds the player have shot. */
    SHOTS,
    /** How many of the shots have hit an enemy. */
    HITS,
    /** Number of killed enemies */
    KILLS,
    /** Number of killed civillians */
    CIVILIANS,
    /** Number of things we keep the statistics */
    NUM_OF_ITEMS
  };
  
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	Statistics();


	/** Destructor
   */
	virtual ~Statistics();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Statistics
   */
  Statistics( const Statistics& rO );

	/** Assignment operator
   * @param     rO                Reference to another Statistics
   * @return    Reference to us.
   */
  Statistics& operator = ( const Statistics& rO );

public:


  ///
  /// Methods
  /// =======

  /** Reads the statistics from given stream. This method expects that the
   * opening tag has been read away from the stream.
   * @param     rIn               Input stream
   * @return    Nonzero if fails.
   */
  int readFrom( std::istream& rIn );
  
  /** Writes the statistics to given stream. This method writes the opening
   * and closing tags too.
   * @param     rOut              Output stream
   * @return    Nonzero if fails
   */
  int writeTo( std::ostream& rOut ) const;


  /** Prepares for the new level. Resets the level statistics
   */
  void resetLevelStats();
  
  /** Resets all the statistics
   */
  void resetAllStats();
  
  /** Updates the timer that calculates the time how long player have been
   * playing the level.
   */
  void updateLevelTimer();
  

  /** Updates the statistics and scores.
   * @param     stat              Which event happened.
   * @param     oid               Type of the object. This is needed when we
   *                              update the for instance the killing scores.
   */
  void updateStatistics( StatType stat, ObjectID::Type oid );
  
  /** Grants given amount of scores
   * @param     scores            Amount of scores
   */
  void grantScores( int scores );
  

  ///
  /// Getter methods
  /// ==============

  /** Returns the current total score count
   * @return    Score count
   */
  int score() const;
  
  /** Returns the requested statistics for current level.
   * @param     stat              Which statistics should be returned.
   * @return    The value.
   */
  int getLevelStats( StatType stat ) const;
  
  /** Returns the requested overall statistics.
   * @param     stat              Which statistics should be returned.
   * @return    The value.
   */
  int getOverallStats( StatType stat ) const;
  
  
  /** Returns the leveltimer value
   * @return    How long we've been playing the current level.
   */
  int levelTimer() const;

protected:

  /** Grants the killing scores
   * @param     oid               Type of the killed object.
   */
  void grantKillingScore( ObjectID::Type oid );



  ///
  /// Members
  /// =======

  /** The total score at the moment */
  int m_score;
  
  /** Calculates how long the player have been playing this level. */
  int m_levelTimer;
  
  /** The statistics about current level. Use <code>enum StatType</code>
   * enumerated constants as indices.
   */
  int m_levelstats[ NUM_OF_ITEMS ];
  
  /** Overall statistics. Use <code>enum StatType</code>
   * enumerated constants as indices.
   */
  int m_overallstats[ NUM_OF_ITEMS ];
  

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: statistics.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:17+02  lamminsa
 * Initial revision
 *
 */
 
